/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.bcel.classfile;

import java.io.DataInput;
import java.io.DataOutputStream;
import java.io.IOException;

import org.apache.bcel.Const;

/**
 * This class represents an entry in the requires table of the Module attribute. Each entry describes a module on which
 * the parent module depends.
 *
 * @see Module
 * @since 6.4.0
 */
public final class ModuleRequires implements Cloneable, Node {

    private final int requiresIndex; // points to CONSTANT_Module_info
    private final int requiresFlags;
    private final int requiresVersionIndex; // either 0 or points to CONSTANT_Utf8_info

    /**
     * Constructs object from file stream.
     *
     * @param file Input stream.
     * @throws IOException if an I/O Exception occurs in readUnsignedShort.
     */
    ModuleRequires(final DataInput file) throws IOException {
        requiresIndex = file.readUnsignedShort();
        requiresFlags = file.readUnsignedShort();
        requiresVersionIndex = file.readUnsignedShort();
    }

    /**
     * Called by objects that are traversing the nodes of the tree implicitly defined by the contents of a Java class.
     * I.e., the hierarchy of methods, fields, attributes, etc. spawns a tree of objects.
     *
     * @param v Visitor object.
     */
    @Override
    public void accept(final Visitor v) {
        v.visitModuleRequires(this);
    }

    /**
     * @return deep copy of this object.
     */
    public ModuleRequires copy() {
        try {
            return (ModuleRequires) clone();
        } catch (final CloneNotSupportedException e) {
            // TODO should this throw?
        }
        return null;
    }

    /**
     * Dumps table entry to file stream in binary format.
     *
     * @param file Output file stream.
     * @throws IOException if an I/O Exception occurs in writeShort.
     */
    public void dump(final DataOutputStream file) throws IOException {
        file.writeShort(requiresIndex);
        file.writeShort(requiresFlags);
        file.writeShort(requiresVersionIndex);
    }

    /**
     * Gets the module name from the constant pool.
     *
     * @param constantPool Array of constants usually obtained from the ClassFile object.
     * @return module name.
     * @since 6.10.0
     */
    public String getModuleName(final ConstantPool constantPool) {
        return constantPool.constantToString(requiresIndex, Const.CONSTANT_Module);
    }

    /**
     * Gets the flags for this ModuleRequires.
     *
     * @return the requiresFlags.
     * @since 6.10.0
     */
    public int getRequiresFlags() {
        return requiresFlags;
    }

    /**
     * Gets the required version from the constant pool.
     *
     * @param constantPool Array of constants usually obtained from the ClassFile object.
     * @return required version, "0" if version index is 0.
     * @since 6.10.0
     */
    public String getVersion(final ConstantPool constantPool) {
        return requiresVersionIndex == 0 ? "0" : constantPool.getConstantString(requiresVersionIndex, Const.CONSTANT_Utf8);
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        return "requires(" + requiresIndex + ", " + String.format("%04x", requiresFlags) + ", " + requiresVersionIndex + ")";
    }

    /**
     * @return Resolved string representation.
     */
    public String toString(final ConstantPool constantPool) {
        final StringBuilder buf = new StringBuilder();
        final String moduleName = getModuleName(constantPool);
        buf.append(moduleName);
        buf.append(", ").append(String.format("%04x", requiresFlags));
        final String version = getVersion(constantPool);
        buf.append(", ").append(version);
        return buf.toString();
    }
}
